#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "image.h"
#include "pgm.h"
#include "ppm.h"

typedef struct fnc_list_t {
    char *name;                      /* A függvény neve szövegesen  */
    void (*fnc)();                   /* A függvényre mutató pointer */
} fnc_list_t;

static void  help();
static void all_transformations();

fnc_list_t pgm_functions[] = {
    { "inverse",   pgm_inverse},
    { "flip", pgm_flip},
    { "all", all_transformations},
    { "help",  help  },
    {  NULL,   NULL  }                          /* A táblázat végét jelzi */
};

fnc_list_t ppm_functions[] = {
    { "inverse",   ppm_inverse},
    { "flip", ppm_flip},
    { "all", all_transformations}, 
    { "help",  help  },
    {  NULL,   NULL  }                          /* A táblázat végét jelzi */
};

image* p_read(const char* filename, const char* ext) {
    image* img = NULL;
    if (!strncmp(ext, ".pgm", 4)) {
        printf("Reading image PGM\n");
        img = (image*) malloc (sizeof(image));
        img->type = PGM;
        pgm_read(filename, &(img->image.pgm));
    } else if (!strncmp(ext, ".ppm", 4)) {
        printf("Reading image PPM\n");
        img = (image*) malloc (sizeof(image));
        img->type = PPM;
        ppm_read(filename, &(img->image.ppm));
    }
    return img;
}

void p_write(const char* filename, image *image) {
    if (image==NULL)
        return;
    if (image->type == PGM) {
        printf("Writing PGM\n");
        pgm_write(filename, &(image->image.pgm));
    }   else {
        printf("Writing PPM\n");
        ppm_write(filename, &(image->image.ppm));
    }
}

void help(fnc_list_t *fp) {
    printf("The implemented functions:\n");
    for(;fp->name; ++fp) {
        printf("\t %s\n", fp->name);
    }
    exit(EXIT_SUCCESS);
}

void all_transformations(fnc_list_t *fp, image *image) {
  for(;fp->name; ++fp) {
    if (fp->fnc != help && fp->fnc != all_transformations) 
       (*fp->fnc)(&(image->image));
  } 
}

void p_transform(char *fnc_name, image *image) {
    if (image==NULL) return;
    fnc_list_t *fp;
    fnc_list_t *all=NULL;
    if (image->type == PGM) {
       fp = pgm_functions;
    } else if (image->type == PPM) {
       fp = ppm_functions;
    }
    all=fp;
    for(;fp->name && strcmp(fp->name, fnc_name);fp++);

    if (fp->name == NULL) {
      printf("Unimplemented transformation!\n");
      exit(EXIT_SUCCESS);
    }

    if (fp->fnc == help) { 
       help(all);
    }

    if (fp->fnc == all_transformations) {
      all_transformations(all, image);
    } else 
       (*fp->fnc)(&(image->image)); //a kivalasztott fuggveny meghivasa
}

void p_free(image *image) {
    if (image == NULL)
        return;
    if (image->type == PGM)
        pgm_free(&(image->image.pgm));
    else
        ppm_free(&(image->image.ppm));
    free(image);
}
